//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

/*!
\qmltype     UiTabBar
\inqmlmodule we.captivo.Ui
\since       we.captivo.Ui 1.0
\inherits    QtQuick2::FocusScope
\ingroup     captivo-ui-controls

\brief A horizontal tab bar.

The UiTabBar component displays a horizontal row of tabs with each tab
labelled with a text derived from the model data.
*/

FocusScope {
    id : uiTabBar

    /*!
    This signal is emitted when a tab in the UiTabBar is clicked
    or triggered by keyboard interaction, regardless of the tab being active
    already. The \a index of the clicked tab is supplied as argument.
    */
    signal clicked(int index)

    /*!
    \qmlproperty any UiTabBar::model
    \brief The data model to display as tabs.

    The model can be any value accepted as model by a \l {Repeater}.
    The tab label is derived from the model data in the following order:

    \list A
    \li The property \e text provided by the model.
    \li The model data converted with \e toString.
    \li The tab \e index.
    \endlist

    In addition the model-supplied property \e disabled can be used to
    disable a single tab in a tab bar.

    The following examples show several options to use a UiTabBar:

    \qml
    import QtQuick 2.4

    import wesual.Ui 1.0

    UiTabBar {
        model : [ "Tab 1", "Tab 2", "Tab 3" ]
    }
    \endqml

    \qml
    import QtQuick 2.4

    import wesual.Ui 1.0

    UiTabBar {
        model : ListModel {
            ListElement {
                text : "Tab 1"
            }
            ListElement {
                text : "Tab 2"
                disabled : true
            }
        }
    }
    \endqml

    \qml
    import QtQuick 2.4

    import wesual.Ui 1.0

    UiTabBar {
        model : 10
    }
    \endqml
    */
    property alias model : repeater.model

    /*!
    \brief The index of the currently selected tab.

    \nobinding
    */
    property int currentIndex : -1

    /*!
    \internal
    */
    property Component backgroundDelegate : Item {
        Rectangle {
            color  : UiColors.getColor(UiColors.Black, 0.1)
            height : 1
            anchors {
                bottom : parent.bottom
                left   : parent.left
                right  : parent.right
            }
        }
    }

    /*!
    \internal
    */
    property Component highlightDelegate : Item {
        Rectangle {
            color  : UiColors.getColor(UiColors.ButtonNormalBlue, 0.9)
            height : 1
            anchors {
                bottom : parent.bottom
                left   : parent.left
                right  : parent.right
            }
        }
    }

    width  : 250
    height : 24

    activeFocusOnTab : true
    onWidthChanged   : p_.layout();

    onActiveFocusChanged : {
        if (activeFocus && repeater.itemAt(currentIndex)) {
            p_.focusedTabIndex = currentIndex;
        }
    }

    Component.onCompleted : {
        if (uiTabBar.backgroundDelegate) {
            var bg = uiTabBar.backgroundDelegate.createObject(uiTabBar);
            bg.z = -1;
            bg.anchors.fill = uiTabBar;
        }
        marker.activeItem = repeater.itemAt(currentIndex);
    }
    onCurrentIndexChanged : marker.activeItem = repeater.itemAt(currentIndex)

    Keys.onLeftPressed : {
        // Find tab left of current tab
        for (var i = p_.focusedTabIndex - 1; i >= 0; --i) {
            var item = repeater.itemAt(i);
            if (item && item.enabled) {
                p_.focusedTabIndex = i;
                return;
            }
        }
    }
    Keys.onRightPressed : {
        // Find tab right of current tab
        for (var i = p_.focusedTabIndex + 1; i < repeater.count; ++i) {
            var item = repeater.itemAt(i);
            if (item && item.enabled) {
                p_.focusedTabIndex = i;
                return;
            }
        }
    }
    Keys.onSpacePressed  : p_.triggerFocusTab()
    Keys.onEnterPressed  : p_.triggerFocusTab()
    Keys.onReturnPressed : p_.triggerFocusTab()

    QtObject {
        id : p_

        property int focusedTabIndex : -1

        readonly property int minimumTabWidth : 24

        readonly property int tabSpacing : 18

        readonly property Item focusedTab : repeater.itemAt(focusedTabIndex)
        readonly property Item activeTab  :
            repeater.itemAt(uiTabBar.currentIndex)

        function triggerFocusTab() {
            uiTabBar.currentIndex = p_.focusedTabIndex;
            uiTabBar.clicked(uiTabBar.currentIndex);
        }

        function layout() {
            var avail = uiTabBar.width;

            var tabWidths = [];
            var items = [];
            var sum = 0;
            for (var i = 0; i < repeater.count; ++i) {
                var item = repeater.itemAt(i);
                if (item) {
                    var iw = Math.max(minimumTabWidth, item.implicitWidth);
                    tabWidths.push(iw);
                    items.push(item);
                    sum += iw;
                }
            }
            sum += Math.max(0, repeater.count - 1) * p_.tabSpacing;
            if (sum > uiTabBar.width) {
                var overflow = Math.ceil(sum - uiTabBar.width);

                while (overflow > 0) {
                    var maxIdx = -1;
                    var max = 0;
                    for (i = 0; i < tabWidths.length; ++i) {
                        if (tabWidths[i] > max) {
                            max = tabWidths[i];
                            maxIdx = i;
                        }
                    }
                    tabWidths[maxIdx] -= 1;
                    --overflow;
                }
            }

            for (i = 0; i < items.length; ++i) {
                items[i].width = tabWidths[i];
            }
        }
    }

    Item {
        id : marker

        property Item activeItem

        anchors.bottom : parent.bottom
        height : parent.height
        width  : activeItem ? activeItem.width : 0
        z      : 0
        x      : activeItem ? (activeItem.x + tabs.x) : 0

        Component.onCompleted : {
            if (uiTabBar.highlightDelegate) {
                var hl = uiTabBar.highlightDelegate.createObject(marker);
                hl.anchors.fill = marker;
            }
        }

        Behavior on width {
            NumberAnimation {
                duration    : 100
                easing.type : Easing.InOutSine
            }
        }
        Behavior on x {
            NumberAnimation {
                duration    : 100
                easing.type : Easing.InOutSine
            }
        }
    }

    UiFocusRect {
        x      : p_.focusedTab ? (p_.focusedTab.x - 2)  : 0
        y      : p_.focusedTab ? (p_.focusedTab.y - 1)  : 0
        width  : p_.focusedTab ? (p_.focusedTab.width + 4)  : 0
        height : 18
        target : uiTabBar
    }

    Row {
        id : tabs

        spacing : p_.tabSpacing

        Repeater {
            id : repeater

            onItemAdded : {
                if (index === 0 && uiTabBar.currentIndex === -1) {
                    uiTabBar.currentIndex = 0;
                }
                p_.layout();
            }
            onItemRemoved : {
                if (uiTabBar.currentIndex >= count) {
                    for (var i = count - 1; i >= 0; --i) {
                        var item = itemAt(i);
                        if (item && item.enabled) {
                            uiTabBar.currentIndex = i;
                            return;
                        }
                    }
                    uiTabBar.currentIndex = -1;
                }

                p_.layout();
            }

            delegate : Item {
                id : tab

                readonly property bool active :
                    index === uiTabBar.currentIndex

                enabled : !(model.disabled === true)
                height  : uiTabBar.height
                implicitWidth : label.implicitWidth

                Text {
                    id : label

                    text  :  {
                        // Determine tab text from
                        var text;
                        if (typeof(model.text) !== "undefined") {
                            return model.text;
                        } else {
                            return modelData.toString();
                        }
                    }

                    width : parent.width
                    elide : Text.ElideRight
                    color : UiColors.getColor(UiColors.MediumGrey33)
                    font  : UiFonts.getFont(UiFonts.RegularCapitalized, 13)
                    horizontalAlignment : Qt.AlignHCenter
                }
                MouseArea {
                    id : tabMouseArea

                    anchors.fill : parent
                    cursorShape  : Qt.PointingHandCursor
                    hoverEnabled : tab.enabled
                    onClicked    : {
                        uiTabBar.clicked(index);
                        uiTabBar.currentIndex = index;
                    }
                }

                states : [
                    State {
                        name : "disabled"
                        when : !enabled

                        PropertyChanges {
                            target : label
                            color: UiColors.getColor(UiColors.MediumLightGrey27)
                        }
                        PropertyChanges {
                            target : tabMouseArea
                            cursorShape : Qt.ArrowCursor
                        }
                    },
                    State {
                        name : "active"
                        when : tab.active

                        PropertyChanges {
                            target : label
                            color : UiColors.getColor(UiColors.ButtonNormalBlue)
                        }
                    },
                    State {
                        name : "hovered"
                        when : tabMouseArea.containsMouse

                        PropertyChanges {
                            target : label
                            color  : UiColors.getColor(UiColors.ButtonHoverBlue)
                        }
                    }
                ]

                transitions : Transition {
                    ColorAnimation {
                        property : "color"
                        duration : 70
                    }
                }
            }
        }
    }
}
